/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.core.runtimeservices.homepage;

import java.util.Optional;

import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.inject.Named;

import org.springframework.stereotype.Service;

import org.apache.causeway.applib.annotation.HomePage;
import org.apache.causeway.applib.annotation.PriorityPrecedence;
import org.apache.causeway.applib.services.factory.FactoryService;
import org.apache.causeway.applib.services.homepage.HomePageResolverService;
import org.apache.causeway.core.config.beans.CausewayBeanTypeRegistry;
import org.apache.causeway.core.metamodel.commons.ClassExtensions;
import org.apache.causeway.core.runtimeservices.CausewayModuleCoreRuntimeServices;

/**
 * Default implementation of {@link HomePageResolverService}, which uses the view model annotated with
 * {@link HomePage} as the home page.
 *
 * <p>
 *     If there is more than one such view model so annotated, then the one chosen is not defined
 *     (in effect, can be considered is randomly chosen).
 * </p>
 *
 * @since 2.0 {@index}
 */
@Service
@Named(CausewayModuleCoreRuntimeServices.NAMESPACE + ".HomePageResolverServiceDefault")
@jakarta.annotation.Priority(PriorityPrecedence.MIDPOINT)
public class HomePageResolverServiceDefault implements HomePageResolverService {

    private final FactoryService factoryService;
    private final CausewayBeanTypeRegistry causewayBeanTypeRegistry;

    private Optional<Class<?>> viewModelTypeForHomepage;

    @Inject
    public HomePageResolverServiceDefault(
            final FactoryService factoryService,
            final CausewayBeanTypeRegistry causewayBeanTypeRegistry) {

        this.factoryService = factoryService;
        this.causewayBeanTypeRegistry = causewayBeanTypeRegistry;
    }

    @PostConstruct
    public void init() {
        viewModelTypeForHomepage = causewayBeanTypeRegistry.findHomepageViewmodel();
    }

    @Override
    public Object getHomePage() {
        return viewModelTypeForHomepage
                .map(ClassExtensions::newInstance)
                .map(factoryService::viewModel)
                .orElse(null);
    }

}
