/* -*-pgsql-c-*- */
/*
 * $Header: /home/t-ishii/repository/pgpool/pool_process_query.c,v 1.6 2004/02/05 14:27:44 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * pool_process_query.c: query processing stuff
 *
*/
#include "config.h"
#include <errno.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include <stdlib.h>
#include <unistd.h>

#include "pool.h"

static POOL_STATUS NotificationResponse(POOL_CONNECTION *frontend, 
										POOL_CONNECTION *backend);

static POOL_STATUS Query(POOL_CONNECTION *frontend, 
						 POOL_CONNECTION *backend);

static POOL_STATUS ReadyForQuery(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION *backend, int send_ready);

static POOL_STATUS CompleteCommandResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION *backend);

static POOL_STATUS CopyInResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION *backend);

static POOL_STATUS CopyOutResponse(POOL_CONNECTION *frontend, 
								   POOL_CONNECTION *backend);

static POOL_STATUS CopyDataRows(POOL_CONNECTION *src,
								POOL_CONNECTION *dest);

static POOL_STATUS CursorResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION *backend);

#ifdef NOTUSED
static POOL_STATUS CancelRequest(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION *backend);
#endif

static POOL_STATUS NoticeResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION *backend);

static POOL_STATUS EmptyQueryResponse(POOL_CONNECTION *frontend,
									  POOL_CONNECTION *backend);

static int RowDescription(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION *backend);

static POOL_STATUS AsciiRow(POOL_CONNECTION *frontend, 
							POOL_CONNECTION *backend,
							short num_fields);

static POOL_STATUS BinaryRow(POOL_CONNECTION *frontend, 
							 POOL_CONNECTION *backend,
							 short num_fields);

static POOL_STATUS FunctionCall(POOL_CONNECTION *frontend, 
								POOL_CONNECTION *backend);

static POOL_STATUS FunctionResultResponse(POOL_CONNECTION *frontend, 
										  POOL_CONNECTION *backend);

static POOL_STATUS ProcessFrontendResponse(POOL_CONNECTION *frontend, 
											POOL_CONNECTION *backend);

POOL_STATUS pool_process_query(POOL_CONNECTION *frontend, 
							   POOL_CONNECTION *backend,
							   int connection_reuse)
{
	char kind;	/* packet kind (backend) */
	char fkind;	/* packet kind (frontend) */
	short num_fields = 0;
	fd_set	readmask;
	fd_set	writemask;
	fd_set	exceptmask;
	int fds;
	POOL_STATUS status;

	if (connection_reuse)
	{
		status = ReadyForQuery(frontend, backend, 0);
		if (status != POOL_CONTINUE)
			return status;
	}

	for (;;)
	{
		kind = 0;
		fkind = 0;
		if (backend->len == 0 && frontend->len == 0)
		{

			struct timeval timeout;

			timeout.tv_sec = 1;
			timeout.tv_usec = 0;

			FD_ZERO(&readmask);
			FD_ZERO(&writemask);
			FD_ZERO(&exceptmask);
			FD_SET(frontend->fd, &readmask);
			FD_SET(backend->fd, &readmask);
			FD_SET(frontend->fd, &exceptmask);
			FD_SET(backend->fd, &exceptmask);

#ifdef DEBUG
			pool_debug("pid: %d select in child", getpid());
#endif
			fds = select(Max(frontend->fd, backend->fd)+1,
						 &readmask, &writemask, &exceptmask, NULL);

			if (fds == -1)
			{
				if (errno == EINTR)
					continue;

				pool_error("select() failed. reason: %s",strerror(errno));
				return POOL_FATAL;
			}

			if (fds == 0)
			{
				return POOL_CONTINUE;
			}

			if (FD_ISSET(backend->fd, &readmask))
			{
				pool_read(backend, &kind, 1);
				pool_debug("read kind from backend %c", kind);
			}

			if (FD_ISSET(frontend->fd, &exceptmask))
			{
				return POOL_END;
			}
			if (FD_ISSET(backend->fd, &exceptmask))
			{
				return POOL_FATAL;
			}

			if (FD_ISSET(frontend->fd, &readmask))
			{
				status = ProcessFrontendResponse(frontend, backend);
				if (status != POOL_CONTINUE)
					return status;

				continue;
			}
		}
		else
		{
			if (backend->len > 0)
			{
				pool_read(backend, &kind, 1);
				pool_debug("read kind from backend pending data %c len: %d po: %d", kind, backend->len, backend->po);
			}
			if (frontend->len > 0)
			{
				status = ProcessFrontendResponse(frontend, backend);
				if (status != POOL_CONTINUE)
					return status;

				continue;
			}
		}

		/* Prrocess backend Response */
		switch (kind)
		{
			case 'A':
				/* Notification  response */
				status = NotificationResponse(frontend, backend);
				break;

			case 'B':
				/* BinaryRow */
				status = BinaryRow(frontend, backend, num_fields);
				break;

			case 'C':
				/* Complete command response */
				status = CompleteCommandResponse(frontend, backend);
				break;

			case 'D':
				/* AsciiRow */
				status = AsciiRow(frontend, backend, num_fields);
				break;

			case 'E':
				/* Error Response */
				status = ErrorResponse(frontend, backend);
				break;

			case 'G':
				/* CopyIn Response */
				status = CopyInResponse(frontend, backend);
				break;

			case 'H':
				/* CopyOut Response */
				status = CopyOutResponse(frontend, backend);
				break;

			case 'I':
				/* Empty Query Response */
				status = EmptyQueryResponse(frontend, backend);
				break;

			case 'N':
				/* Notice Response */
				status = NoticeResponse(frontend, backend);
				break;

			case 'P':
				/* CursorResponse */
				status = CursorResponse(frontend, backend);
				break;

			case 'T':
				/* RowDescription */
				status = RowDescription(frontend, backend);
				if (status < 0)
					return POOL_ERROR;

				num_fields = status;
				status = POOL_CONTINUE;
				break;

			case 'V':
				/* FunctionResultResponse and FunctionVoidResponse */
				status = FunctionResultResponse(frontend, backend);
				break;

			case 'Z':
				/* Ready for query */
				status = ReadyForQuery(frontend, backend, 1);
				break;
				
			default:
				pool_error("Unknown message type %c(%02x)", kind, kind);
				exit(1);
		}
		if (status != POOL_CONTINUE)
			return status;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS Query(POOL_CONNECTION *frontend, 
						 POOL_CONNECTION *backend)
{
	char kind = 'Q';
	char *string;
	int len;

/* read actual query */
	string = pool_read_string(frontend, &len, 0);
	if (string == NULL)
		return POOL_END;

/* forward the query to the backend */
	pool_write(backend, &kind, 1);
	if (pool_write_and_flush(backend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS ReadyForQuery(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION *backend, int send_ready)
{
	char kind;

	if (send_ready)
	{
		kind = 'Z';
		if (pool_write_and_flush(frontend, &kind, 1) < 0)
		{
			return POOL_END;
		}
	}

	return ProcessFrontendResponse(frontend, backend);
}

static POOL_STATUS CompleteCommandResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION *backend)
{
	char *string;
	int len;

/* read command tag */
	string = pool_read_string(backend, &len, 0);
	if (string == NULL)
		return POOL_END;

/* forward to the frontend */
	pool_write(frontend, "C", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static int RowDescription(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION *backend)
{
	short num_fields;
	int oid, mod;
	short size;
	char *string;
	int len;
	int i;

/* # of fields (could be 0) */
	pool_read(backend, &num_fields, sizeof(short));
/* forward it to the frontend */
	pool_write(frontend, "T", 1);
	pool_write(frontend, &num_fields, sizeof(short));

	num_fields = htons(num_fields);
	for (i = 0;i<num_fields;i++)
	{
		/* field name */
		string = pool_read_string(backend, &len, 0);
		if (string == NULL)
			return POOL_END;

		pool_write(frontend, string, len);

		/* oid */
		pool_read(backend, &oid, sizeof(int));
		pool_write(frontend, &oid, sizeof(int));

		/* size */
		pool_read(backend, &size, sizeof(short));
		pool_write(frontend, &size, sizeof(short));

		/* modifier */
		pool_read(backend, &mod, sizeof(int));
		pool_write(frontend, &mod, sizeof(int));
	}

	if (pool_flush(frontend) < 0)
	{
		return POOL_END;
	}

	return num_fields;
}

static POOL_STATUS AsciiRow(POOL_CONNECTION *frontend, 
							POOL_CONNECTION *backend,
							short num_fields)
{
	static char nullmap[8192];
	int nbytes;
	int i;
	unsigned char mask;
	int size;
	char *buf;
	char msgbuf[1024];

	pool_write(frontend, "D", 1);

	nbytes = (num_fields + 7)/8;

	if (nbytes <= 0)
		return POOL_CONTINUE;

	/* NULL map */
	pool_read(backend, nullmap, nbytes);
	if (pool_write_and_flush(frontend, nullmap, nbytes) < 0)
	{
		return POOL_END;
	}

	mask = 0;

	for (i = 0;i<num_fields;i++)
	{
		if (mask == 0)
			mask = 0x80;

		/* NOT NULL? */
		if (mask & nullmap[i/8])
		{
			/* field size */
			if (pool_read(backend, &size, sizeof(int)) < 0)
				return POOL_END;

			pool_write(frontend, &size, sizeof(int));
			size = htonl(size) - 4;

			/* read and send actual data only when size > 0 */
			if (size > 0)
			{
				buf = malloc(size);
				if (buf == NULL)
				{
					pool_error("AsciiRow: out of memory");
					return POOL_END;
				}
				/* actual data */
				if (pool_read(backend, buf, size) < 0)
				{
					free(buf);
					return POOL_END;
				}
				pool_write(frontend, buf, size);
				snprintf(msgbuf, Min(sizeof(msgbuf), size), "%s", buf);
				free(buf);
			}
			else
			{
				*msgbuf = '\0';
			}
			pool_debug("AsciiRow: len:%d data: %s", size, msgbuf);
		}
		mask >>= 1;
	}

	if (pool_flush(frontend) < 0)
	{
		return POOL_END;
	}

	return POOL_CONTINUE;
}

static POOL_STATUS BinaryRow(POOL_CONNECTION *frontend, 
							 POOL_CONNECTION *backend,
							 short num_fields)
{
	static char nullmap[8192];
	int nbytes;
	int i;
	unsigned char mask;
	int size;
	char *buf;

	pool_write(frontend, "B", 1);

	nbytes = (num_fields + 7)/8;

	if (nbytes <= 0)
		return POOL_CONTINUE;

	/* NULL map */
	pool_read(backend, nullmap, nbytes);
	if (pool_write_and_flush(frontend, nullmap, nbytes) < 0)
	{
		return POOL_END;
	}

	mask = 0;

	for (i = 0;i<num_fields;i++)
	{
		if (mask == 0)
			mask = 0x80;

		/* NOT NULL? */
		if (mask & nullmap[i/8])
		{
			/* field size */
			pool_read(backend, &size, sizeof(int));
			pool_write(frontend, &size, sizeof(int));
			size = htonl(size);
			buf = malloc(size);
			if (buf == NULL)
			{
				pool_error("BinaryRow: out of memory");
				return POOL_END;
			}
			/* actual data */
			pool_read(backend, buf, size);
			pool_write(frontend, buf, size);
			free(buf);
		}
		mask >>= 1;
	}

	if (pool_flush(frontend) < 0)
	{
		return POOL_END;
	}

	return POOL_CONTINUE;
}

static POOL_STATUS CursorResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION *backend)
{
	char *string;
	int len;

/* read cursor name */
	string = pool_read_string(backend, &len, 0);
	if (string == NULL)
		return POOL_END;

/* forward to the frontend */
	pool_write(frontend, "P", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

POOL_STATUS ErrorResponse(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION *backend)
{
	char *string;
	int len;

/* read error message */
	string = pool_read_string(backend, &len, 0);
	if (string == NULL)
		return POOL_END;

/* forward to the frontend */
	pool_write(frontend, "E", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS NoticeResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION *backend)
{
	char *string;
	int len;

/* read notice message */
	string = pool_read_string(backend, &len, 0);
	if (string == NULL)
		return POOL_END;

/* forward to the frontend */
	pool_write(frontend, "N", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS CopyInResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION *backend)
{
/* forward to the frontend */
	if (pool_write_and_flush(frontend, "G", 1) < 0)
	{
		return POOL_END;
	}
	return CopyDataRows(frontend, backend);
}

static POOL_STATUS CopyOutResponse(POOL_CONNECTION *frontend, 
								   POOL_CONNECTION *backend)
{
/* forward to the frontend */
	if (pool_write_and_flush(frontend, "H", 1) < 0)
	{
		return POOL_END;
	}
	return CopyDataRows(backend, frontend);
}

static POOL_STATUS CopyDataRows(POOL_CONNECTION *src,
								POOL_CONNECTION *dest)
{
	char *string;
	int len;

#ifdef DEBUG
	int i = 0;
	char buf[1024];
#endif

	for (;;)
	{
		string = pool_read_string(src, &len, 1);
		if (string == NULL)
			return POOL_END;

#ifdef DEBUG
		strncpy(buf, string, len);
		pool_debug("copy line %d %d bytes :%s:", i++, len, buf);
#endif

		pool_write(dest, string, len);

		if (len == 3)
		{
			/* end of copy? */
			if (string[0] == '\\' &&
				string[1] == '.' &&
				string[2] == '\n')
			{
				break;
			}
		}
	}

	if (pool_flush(dest) < 0)
		return POOL_END;

	return POOL_CONTINUE;
}

#ifdef NOTUSED
static POOL_STATUS CancelRequest(POOL_CONNECTION *frontend,
								 POOL_CONNECTION *backend)
{
	int packet_size;
	int request_code;
	int pid;
	int secret_key;

	pool_write(backend, "F", 1);

	if (pool_read(frontend, &packet_size, sizeof(packet_size) < 0))
		return POOL_ERROR;
	pool_write(backend, &packet_size, sizeof(packet_size));

	if (pool_read(frontend, &request_code, sizeof(request_code) < 0))
		return POOL_ERROR;
	pool_write(backend, &request_code, sizeof(request_code));

	if (pool_read(frontend, &pid, sizeof(pid) < 0))
		return POOL_ERROR;
	pool_write(backend, &pid, sizeof(pid));

	if (pool_read(frontend, &secret_key, sizeof(secret_key) < 0))
		return POOL_ERROR;
	pool_write(backend, &secret_key, sizeof(secret_key));

	return pool_flush(backend);
}
#endif

static POOL_STATUS EmptyQueryResponse(POOL_CONNECTION *frontend,
									  POOL_CONNECTION *backend)
{
	char c;

	if (pool_read(backend, &c, sizeof(c)) < 0)
		return POOL_END;

	pool_write(frontend, "I", 1);
	return pool_write_and_flush(frontend, "", 1);
}

static POOL_STATUS NotificationResponse(POOL_CONNECTION *frontend, 
										POOL_CONNECTION *backend)
{
	int pid;
	char *condition;
	int len;

	pool_write(frontend, "A", 1);

	if (pool_read(backend, &pid, sizeof(pid)) < 0)
		return POOL_ERROR;

	condition = pool_read_string(backend, &len, 0);
	if (condition == NULL)
		return POOL_END;

	pool_write(frontend, &pid, sizeof(pid));

	return pool_write_and_flush(frontend, condition, len);
}

static POOL_STATUS FunctionCall(POOL_CONNECTION *frontend, 
								POOL_CONNECTION *backend)
{
	char dummy[2];
	int oid;
	int argn;
	int i;

	pool_write(backend, "F", 1);

	/* dummy */
	if (pool_read(frontend, dummy, sizeof(dummy)) < 0)
		return POOL_ERROR;
	pool_write(backend, dummy, sizeof(dummy));

	/* function object id */
	if (pool_read(frontend, &oid, sizeof(oid)) < 0)
		return POOL_ERROR;
	pool_write(backend, &oid, sizeof(oid));

	/* number of arguments */
	if (pool_read(frontend, &argn, sizeof(argn)) < 0)
		return POOL_ERROR;
	pool_write(backend, &argn, sizeof(argn));

	argn = htonl(argn);

	for (i=0;i<argn;i++)
	{
		int len;
		char *arg;

		/* length of each argument in bytes */
		if (pool_read(frontend, &len, sizeof(len)) < 0)
			return POOL_ERROR;

		pool_write(backend, &len, sizeof(len));

		len = htonl(len);

		arg = malloc(len);
		if (arg == NULL)
		{
			pool_error("FuncationCall: out of memory");
			return POOL_ERROR;
		}

		/* argument value itself */
		if (pool_read(frontend, arg, len) < 0)
			return POOL_ERROR;
		pool_write(backend, arg, len);
		free(arg);
	}
	return pool_flush(backend);
}

static POOL_STATUS FunctionResultResponse(POOL_CONNECTION *frontend, 
										  POOL_CONNECTION *backend)
{
	char dummy;
	int len;
	char *result;

	pool_write(frontend, "V", 1);

	if (pool_read(backend, &dummy, 1) < 0)
			return POOL_ERROR;
	pool_write(frontend, &dummy, 1);

	/* non empty result? */
	if (dummy == 'G')
	{
		/* length of result in bytes */
		if (pool_read(backend, &len, sizeof(len)) < 0)
			return POOL_ERROR;

		pool_write(frontend, &len, sizeof(len));

		len = htonl(len);

		result = malloc(len);
		if (result == NULL)
		{
			pool_error("FuncationResultResponse: out of memory");
			return POOL_ERROR;
		}

		/* result value itself */
		if (pool_read(backend, result, len) < 0)
			return POOL_ERROR;
		pool_write(frontend, result, len);
		free(result);
	}

	/* unused ('0') */
	if (pool_read(backend, &dummy, 1) < 0)
		return POOL_ERROR;
	pool_write(frontend, "0", 1);

	return pool_flush(frontend);
}

static POOL_STATUS ProcessFrontendResponse(POOL_CONNECTION *frontend, 
											POOL_CONNECTION *backend)
{
	char fkind;
	POOL_STATUS status;

	if (pool_read(frontend, &fkind, 1) < 0)
	{
		pool_error("ProccessFrontendResponse: failed to read kind");
		return POOL_END;
	}

	pool_debug("read kind from frontend %c(%02x)", fkind, fkind);

	switch (fkind)
	{
		case 'X':
			status = POOL_END;
			break;

		case 'Q':
			status = Query(frontend, backend);
			break;

		case 'F':
			status = FunctionCall(frontend, backend);
			break;

		default:
			pool_error("ProcessFrontendResponse: unknown message type %c(%02x)", fkind, fkind);
			status = POOL_ERROR;
			break;
	}

	return status;
}
